﻿#Requires -Version 5.0
#Requires -Modules @{ModuleName="PWPS_DAB";ModuleVersion='1.12.2.0'}
#Requires -RunAsAdministrator

[CmdletBinding()]
param(

    [ValidateNotNullOrEmpty()]
    [Parameter(
        HelpMessage = 'Path to local folder to create report in.',
        Position = 0 )]
    [string] $Path = 'D:\TEMP',

    [Parameter(
        HelpMessage = 'Name of report to generate.',
        Position = 1 )]
    [string] $OutFileName = "PWLicensingReport_$(Get-Date -Format yyyyMMdd).xlsx"

) # end param

BEGIN {

    $Continue = $true
    
    $StartTime = Get-Date
    Write-Verbose -Message "[BEGIN] Start time: $StartTime"
   
    <# Test to see if the provided path exists. If not, set the variable Continue to false.  
        This will be used in the PROCESS code block to exit the script. #> 
    if( -not (Test-Path -Path $Path -PathType Container -Verbose -ErrorAction Stop)) {
        Write-Warning -Message "[BEGIN] Folder '$Path' not found. Update path variable value and try again."
        $Continue = $false
    } else {
        Write-Verbose -Message "[BEGIN] Verified folder '$Path' exists."
    }
   
    <# Determine if we are currently logged into a ProjectWise datasource.  
        If not, display the ProjectWise Login Dialog. #>
    if(Get-PWCurrentDatasource) {
        Write-Verbose -Message "[BEGIN] Currently logged into ProjectWise datasource '$(Get-PWCurrentDatasource)'." -Verbose
    } else {
        if(New-PWLogin -UseGui -Verbose) {
            Write-Verbose -Message "[BEGIN] Successfully logged into ProjectWise datasource '$(Get-PWCurrentDatasource)'." -Verbose
        } else {
            Write-Error -Message '[BEGIN] Failed to log into ProjectWise datasource.'
            $Continue = $false        
        }
    } # end if(Get-PWCurrentDatasource...
    
} # end BEGIN

PROCESS {
    if( -not ($Continue)) {
        Write-Warning -Message "[PROCESS] Exiting script."
    } else { 
        
        #region Create Datatable
        
        <# Create a datatable to return desired data.

                Add entry to datatable for each user.
                UserName
                UserDescription
                LicensingSource
        #>
    
        $dtLicensingData = New-Object System.Data.DataTable ('Licensing_Info')
        $dtLicensingData.Columns.AddRange(@("UserName", "UserDescription", "LicensingSource"))
    
        #endregion Create Datatable
    
    
        #region Getting ProjectWise User Account Information.

        $pwUsers = Get-PWUsersByMatch
    
        # The following is a counter for the progress bar. 
        $Counter_Users = 1
        $ItemCount_Users = $pwUsers.count
        # Loop through each item in the collection.
        foreach ($pwUser in $pwUsers) {
        
            #region PROGRESS SECTION
    
            $Progress_Users = @{ 
                Activity = "Getting user information for '$($pwUser.Name)'." 
                Status = "Processing $Counter_Users of $ItemCount_Users" 
                PercentComplete = $([math]::Round($(($Counter_Users / $ItemCount_Users) * 100 ), 2)) 
            } 
            Write-Progress @Progress_Users -Id 1
        
            # Increment the counter.
            $Counter_Users++
    
            #endregion PROGRESS SECTION
    
            #region Get license info.
        
            $Licensing = Get-PWUserSettingByUser -InputUsers $pwUser -SettingName Admin_Licensing
            if($Licensing.UserSettingValue -match 'client') { 
                Write-Verbose -Message "[PROCESS] User '$($pwUser.Name)' is using their own ProjectWise license." -verbose
            }
        
            #endregion Get license info.
                
            #region Populate Datatable
            
            # Create new datarow.
            $dr = $dtLicensingData.NewRow()
            # Populate datarow.
            $dr.UserName = $pwUser.Name
            $dr.UserDescription = $pwUser.Description
            $dr.LicensingSource = $Licensing.UserSettingValue
            # Add datarow to the datatable.
            $dtLicensingData.Rows.Add($dr)
        
            #endregion Populate Datatable
            start-sleep -Seconds 5
        } # end foreach ($pwUser...
        
        Write-Progress -Activity 'Completed' -Completed
        
        #endregion Getting ProjectWise User Account Information.
        
        
        #region Export datatables to Excel.
        
        # Create new report from datatable contents.
        Write-Verbose -Message "Exporting data to '$Path\$OutFilePathName'."
        New-XLSXWorkbook -InputTables $dtLicensingData -OutputFileName "$Path\$OutFileName"
        
        #endregion Export datatables to Excel.    
        
    } # end if( -not ($Continue...
} # end PROCESS

END {
    $EndTime = Get-Date
    Write-Verbose -Message "[END] It took $([Math]::Round($EndTime.Subtract($StartTime).TotalMinutes, 2)) minutes to complete process." -Verbose

    Write-Verbose -Message '[END] Logging out of ProjectWise.' -Verbose
    Undo-PWLogin

} # END
