﻿# Variables
# ProjectWise Folder to Export documents from.
$FolderPath_Export = 'PowerShell\FilesToExport'
# ProjectWise Folder to Import documents into.
$FolderPath_Import = 'PowerShell\Import'
# Local folder to export to.
$LocalFolder = 'D:\TEMP\Export\PowerShell'
# Spreadsheet to export document metadata to.
$ExportMetadataFile = "$LocalFolder\PowerShell_Metadata_$(Get-Date -Format yyyyMMddHHmmss).xlsx"

#region EXPORT

# Using splat for readability.
$Splat_Export = @{
    ProjectWiseFolder = $FolderPath_Export
    OutputFolder = $LocalFolder
    ExportMetadata = $true
    ExportMetadataFile = $ExportMetadataFile
}
Export-PWDocuments @Splat_Export

#endregion EXPORT

#region IMPORT

# Parameter / values for Import. 
$InputFolder = $LocalFolder 
$ProjectWiseFolderInput = 'PowerShell\Import'

# Using splat for readability.
$Splat_Import = @{
    InputFolder = $InputFolder
    ProjectWiseFolder = $FolderPath_Import
    ExcludeSourceDirectoryFromTargetPath = $true
    ExtensionsOfInterest = '.dgn;.pdf'
}
Import-PWDocuments @Splat_Import

#endregion IMPORT

#region UPDATE ATTRIBUTES

# Import contents of the generated spreadsheet.
$dts = Get-TablesFromXLSXWorkbook -InputFileName $ExportMetadataFile

# You'll want to use the table which matches the Environment name for the folder.
$Environment = (Get-PWFolders -FolderPath $ProjectWiseFolderInput -JustOne | Select-Object Environment).Environment

# Isolate only the datatable which corresponds to the environment.
$dtAttributes = $dts | Where-Object TableName -eq $Environment

# Get a list of column names from the environment associated with the target folders environment.
$Columns = (Get-PWEnvironmentColumns -EnvironmentName $Environment).GetEnumerator() | Select-Object Name

# Loop through each entry in the datatable.
foreach ($row in $dtAttributes) {
    # Get file name to be updated.
    $FileName = Split-Path $row.ProjectWisePath -Leaf

    # Get the ProjectWise document object for the current document.
    $pwDocObject = Get-PWDocumentsBySearch -FolderPath $ProjectWiseFolderInput -FileName $FileName -JustThisFolder
    if( [string]::IsNullOrEmpty($pwDocObject)) {
        Write-Warning -Message "Failed to select document '$FileName'."
        Continue
    }
    
    # Hashtable to populate with Attribute name / value pairs.
    $AttributesToUpdate = @{}   
    # Populate hashtable.
    foreach ($Column in $Columns.Name) {
        # Skip this column.
        if($Column -eq 'a_attrno') {
            Continue
        }

        # Populate hash table with attribute name and values ONLY when a value is included.
        if( -not ([string]::IsNullOrEmpty($row.$Column))) {
            $AttributesToUpdate.Add($Column, $row.$Column)
        }
    } # end foreach ($Column in $Columns...
    
    # Update attribute values for the current document.
    if( -not (Update-PWDocumentAttributes -InputDocuments $pwDocObject -Attributes $AttributesToUpdate -ReturnBoolean)) {
        Write-Warning -Message "Failed to update document attributes for $($pwDocObject.Name)."
    } 
    
} # end foreach ($row in $dtAttributes...


$AttributesToUpdate.GetEnumerator() | Sort-Object Name

#endregion UPDATE ATTRIBUTES