FUNCTION New-PWScanRefFromReport { 
    <#
            .Synopsis
            Used to run scan reference process on documents.
            .DESCRIPTION
            Used to run scan reference process on documents. 
            Document information is obtained from an existing Reference Report. 
            Generated using the Get-PWReferenceReportForFolder cmdlet.
            .EXAMPLE
            Example of how to use this cmdlet

    #>
    [CmdletBinding()]
    param(
        [ValidateNotNullOrEmpty()]
        [Parameter(
                HelpMessage = 'ProjectWise Datasource containing documents to scan.',
                Position = 0,
            Mandatory = $true)]
        [string] $DatasourceName,

        [ValidateNotNullOrEmpty()]
        [Parameter(
                HelpMessage = 'ProjectWise User Name.',
                Position = 1,
            Mandatory = $true)]
        [string] $UserName,

        [ValidateNotNullOrEmpty()]
        [ValidateScript( { Get-ChildItem -Path $_ -File } )]
        [Parameter(
                HelpMessage = 'Text file to get password from.',
                Position = 2,
            Mandatory = $true)]
        [string] $PasswordFilePathName,
        
        [ValidateNotNullOrEmpty()]
        [ValidateScript({ Get-ChildItem -Path $_ -File -Filter '.xlsx' })]
        [Parameter(
                HelpMessage = 'Path and name of reference report.',
                Position = 3,
            Mandatory = $True)]
        [string] $ReportPathName,
		
        [ValidateNotNullOrEmpty()]
        [Parameter(
                   HelpMessage = 'Parent folder where Work Area resides.',
                   Position = 4,
                   Mandatory = $True)]
        [string] $TargetProjectWiseFolder
    ) # end PARAM block 

    BEGIN {
        # Capture current verbose preference setting prior to setting to Continue.
        $VerbosePreferenceSaved = $VerbosePreference
        $VerbosePreference = "Continue"
        
        $CmdletName = $MyInvocation.MyCommand.Name
        $Continue = $true

        $StartTime = Get-Date
        Write-Verbose -Message "[BEGIN] $StartTime - Entering '$CmdletName' Function..."

        #region Import data from Reference report (Excel spreadsheet).
        
        $dts = Get-TablesFromXLSXWorkbook -InputFileName $ReportPathName -Verbose -ErrorAction Stop
        # ONLY working with the first table.
        $dtReferenceData = $dts[0]
        
        if($dtReferenceData.Rows.Count -gt 0) {
            Write-Verbose -Message "[BEGIN] $($dtReferenceData.Rows.Count) rows returned from '$OutputFilePathName'."
        } else {
            Write-Warning -Message "[BEGIN] No data imported from Reference report '$OutputFilePathName'."
        }
        
        #endregion IMPORT data from Reference report (Excel spreadsheet).
              
    } # end BEGIN block 

    PROCESS { 

        if( -not ($Continue)) {
            Write-Warning -Message "[PROCESS] Exiting script."
        } else {

            try{

                #region Get unique parent document path and name combinations.

                # Create an arraylist to store master file path and names.
                $UniqueMasterPathNames = New-Object -TypeName Collections.ArrayList

                # Process each row within the datatable to populate the arraylist.
                foreach($row in $dtReferenceData.Rows) {
                    $currentMasterPathName = "$($row.ParentFolder)\$($row.ParentDocumentName)"
                    if( -not ($UniqueMasterPathNames.Contains($currentMasterPathName))){
                        $UniqueMasterPathNames.Add($currentMasterPathName) | Out-Null
                    }
                } # end foreach($row...

                Write-Verbose -Message "$($UniqueMasterPathNames.Count) master documents to be scanned."

                #endregion Get unique parent document path and name combinations.

                #region Default values for the New-PWScanForReferences function.
                # The Scan for Reference process requires the ProjectWise datasource, user name and password.
                $PWUserPassword = Get-ClearTextFromEncryptedFile -FileName $PasswordFilePathName
                $ScanMode = 'references'
                $Proximity = 'r:1'
                $Order = 'priority;proximity'
                $Applications = "AutoCAD;MicroStation"

                #endregion Default values for the New-PWScanForReferences function.

                #region RUN New-PWScanForReferences cmdlet for each master document within the arraylist.
				
                foreach ($UniqueMasterPathName in $UniqueMasterPathNames) { 
				
                    # Prepending value from TargetProjectWiseFolder to the current master document.
                    $currentMasterDocument = "$TargetProjectWiseFolder\$UniqueMasterPathName"
    
                    #region CREATE string of reference folder path values.
    
                    $parentFolder = Split-Path $UniqueMasterPathName -Parent
                    $masterDocument = Split-Path $UniqueMasterPathName -Leaf
    
                    Write-Verbose -Message "Getting reference data for document '$masterDocument' in folder '$parentFolder'."
    
                    # Select the data associated with the current master document.
                    $currentMasterDocData = $dtReferenceData.Select("ParentFolder = '$parentfolder' AND ParentDocumentName = '$masterDocument'")
      
                    Write-Verbose -Message " - Retrieved $($currentMasterDocData.Count) records."

                    # Get an array of the unique reference folder paths to be used for the Priority setting.
                    $CurrentReferencePaths = New-Object -TypeName Collections.ArrayList
                    $CurrentReferencePaths = $currentMasterDocData.ReferenceFolder| Select-Object -Unique
                    # Prepend each folder path with the target folder and append with a semi-colon.
                    $CurrentReferencePaths = $CurrentReferencePaths | ForEach-Object { "$TargetProjectWiseFolder\$_;" }
                    # Create a string with the reference paths separated by a semi-colon.
                    foreach ($refPath in $CurrentReferencePaths) {
                        $refPaths += $refPath
                    }
    
                    #endregion CREATE string of reference folder path values.
    
                    Write-Verbose -Message "Priority value: $refPaths"

                    #region RUN New-PWScanForReferences

                    $ScanRef = @{
                        DataSourceName = $DatasourceName
                        UserName = $UserName
                        Password = $PWUserPassword
                        ScanMode = $ScanMode
                        MasterDocuments = $currentMasterDocument
                        Priority = $refPaths
                        #Proximity = $Proximity
                        Order = $Order
                        Applications = $Applications
                    }
                    New-PWScanForReferences  @ScanRef -Verbose
    
                    #endregion RUN New-PWScanForReferences

                    # Clean up variables.
                    Remove-Variable refPaths, refPath, currentMasterDocData, masterDocument, parentFolder
                } # end foreach($UniqueMasterPathName...

                #endregion RUN New-PWScanForReferences cmdlet for each master document within the array.

            } catch {

                $Line = $error[0].InvocationInfo.ScriptLineNumber
                $ErrorMessage = $Error[0].Exception.Message
                Write-Warning -Message "[PROCESS] Error Occurred on line: $Line. $ErrorMessage"

            } # end try/catch... 

        } # end if( -not ($Continue.../else... 

    } # end PROCESS block 

    END {
        $EndTime = Get-Date
        Write-Verbose -Message "[END] It took $($EndTime - $StartTime) to complete the process."
        Write-Verbose -Message "[END] $EndTime - Exiting '$CmdletName' Function..."
        # Reset verbose preference.
        $VerbosePreference = $VerbosePreferenceSaved
    } # end END block

} # end FUNCTION New-PWScanRefFromReport...
Export-ModuleMember New-PWScanRefFromReport
