﻿FUNCTION Get-PWParentWorkArea {
    <#
            .SYNOPSIS
            Get-PWParentWorkArea returns the Parent Work Area / Rich Project folder.

            .DESCRIPTION
            Returns the Parent Work Area / Rich Project folder for the provided ProjectWise Document or Folder object within the folder hierarchy.

            .PARAMETER Document
            Starting ProjecWise Document object. 
        
            .PARAMETER Folder
            Starting ProjectWise Folder object.

            .EXAMPLE
            Get-PWParentWorkArea -Document (Show-PWDocumentSelectionDialog)

            .EXAMPLE
            Get-PWParentWorkArea -Folder (Show-PWFolderBrowserDialog)
    #>
    [CmdletBinding()]
    Param (
        [Parameter (
                ParameterSetName='Document', 
            Mandatory = $true)]
        [ValidateNotNullOrEmpty()]
        [PWPS_DAB.CommonTypes+ProjectWiseDocument] $Document,
        
        [Parameter (
                ParameterSetName='FolderPath', 
            Mandatory = $true)]
        [ValidateNotNullOrEmpty()]
        [PWPS_DAB.CommonTypes+ProjectWiseFolder] $Folder
    ) # end Param...
    
    BEGIN{
     
        $CmdletName = $MyInvocation.InvocationName
        $DateStart = Get-Date
        
        Write-Verbose -Message "[BEGIN] '$DateStart' - Entering '$CmdletName' Function..."

    } # end BEGIN...
    
    PROCESS{
        # Empty string to pass error messages to for the try / catch.
        $Message = [string]::Empty
        try {

            $IsRichProject = $false

            if($Folder){
                $FolderID = $Folder.ProjectID
                $IsRichProject = $Folder.IsRichProject
            } else {
                $FolderID = $Document.ProjectID
            }

            if($IsRichProject -eq $false){
                do {
                    $Folder = Get-PWFolders -FolderID $FolderID -JustOne -Verbose:$false    
                    $IsRichProject = $Folder.IsRichProject
                    
                    $FolderID = $Folder.ParentID
                } until (($IsRichProject -eq $true) -or ($FolderID -eq 0))
                       
                if ( $IsRichProject -ne $true ) {
                    $Message = 'No rich project found in folder tree.'
                    throw $Message
                }
            }
           
            Write-Output $Folder
           
        } catch {
            Write-Warning -Message "[PROCESS] $Message"
        }
    } # end PROCESS...
    
    END {
        $DateEnd = Get-Date
        Write-Verbose -Message "[END] $($DateEnd - $DateStart) - Exiting '$CmdletName' Function..."
    } # end END...
} # end FUNCTION  Get-PWParentWorkArea...
Export-ModuleMember -Function Get-PWParentWorkArea
